#include "scheduled_shutdown.h"
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QPushButton>
#include <QProcess>
#include <QDateTime>
#include <QTimer>
#include <QDebug>
#include <QMessageBox> // [NEW] Required for message boxes

// --- Helper Class: Custom SpinBox Wrapper ---
class SpinBoxWrapper : public QWidget {
public:
    SpinBoxWrapper(QSpinBox* spinBox, QWidget* parent = nullptr) : QWidget(parent) {
        // 1. Setup Main Container
        this->setObjectName("wrapper");
        this->setStyleSheet("QWidget#wrapper { "
                            "  border: 1px solid #ccc; "
                            "  background-color: white; "
                            "  border-radius: 4px; "
                            "}");
        this->setFixedSize(100, 34);

        QHBoxLayout* layout = new QHBoxLayout(this);
        layout->setContentsMargins(0, 0, 0, 0);
        layout->setSpacing(0);

        // 2. Setup SpinBox
        spinBox->setParent(this);
        spinBox->setButtonSymbols(QAbstractSpinBox::NoButtons);
        spinBox->setStyleSheet("QSpinBox { border: none; background: transparent; selection-background-color: #0078d4; }");
        spinBox->setFixedHeight(34); // Match wrapper height

        // 3. Setup Button Container
        QWidget* btnContainer = new QWidget(this);
        btnContainer->setFixedWidth(24);
        btnContainer->setStyleSheet("background-color: #f5f5f5; border-left: 1px solid #ccc;");

        QVBoxLayout* btnLayout = new QVBoxLayout(btnContainer);
        btnLayout->setContentsMargins(0, 0, 0, 0);
        btnLayout->setSpacing(0);

        // 4. Create "+" Button
        QPushButton* btnUp = new QPushButton("+", btnContainer);
        btnUp->setCursor(Qt::PointingHandCursor);
        btnUp->setFlat(true);
        // [TWEAKED] Added padding:0 and specific font settings for perfect centering
        btnUp->setStyleSheet("QPushButton { "
                             "  border: none; "
                             "  font-family: Segoe UI, Arial; "
                             "  font-weight: bold; "
                             "  color: #333; "
                             "  font-size: 16px; "
                             "  padding: 0px; "  // Remove default padding
                             "  margin: 0px; "
                             "  text-align: center; "
                             "  padding-bottom: 2px; " // Slight lift to center visually
                             "} "
                             "QPushButton:hover { background-color: #e0e0e0; } "
                             "QPushButton:pressed { background-color: #d0d0d0; }");

        // 5. Create "-" Button
        QPushButton* btnDown = new QPushButton("-", btnContainer);
        btnDown->setCursor(Qt::PointingHandCursor);
        btnDown->setFlat(true);
        // [TWEAKED] Added padding-bottom to lift the dash visually
        btnDown->setStyleSheet("QPushButton { "
                               "  border: none; "
                               "  border-top: 1px solid #ccc; "
                               "  font-family: Segoe UI, Arial; "
                               "  font-weight: bold; "
                               "  color: #333; "
                               "  font-size: 18px; " // Slightly larger for dash
                               "  padding: 0px; "
                               "  margin: 0px; "
                               "  text-align: center; "
                               "  padding-bottom: 4px; " // Lift the dash up to vertical center
                               "} "
                               "QPushButton:hover { background-color: #e0e0e0; } "
                               "QPushButton:pressed { background-color: #d0d0d0; }");

        // 6. Assemble
        btnLayout->addWidget(btnUp);
        btnLayout->addWidget(btnDown);

        layout->addWidget(spinBox);
        layout->addWidget(btnContainer);

        // 7. Logic
        QObject::connect(btnUp, &QPushButton::clicked, spinBox, [spinBox](){
            spinBox->stepBy(1);
        });
        QObject::connect(btnDown, &QPushButton::clicked, spinBox, [spinBox](){
            spinBox->stepBy(-1);
        });
    }
};

ScheduledShutdownWidget::ScheduledShutdownWidget(QWidget *parent)
    : QWidget(parent)
{
    setupUI();

    // Logic Setup
    timer = new QTimer(this);
    updateTimeInfo();
    connect(timer, &QTimer::timeout, this, &ScheduledShutdownWidget::updateTimeInfo);
    timer->start(1000);

    // Connect spinbox changes
    connect(hourSpinBox, &QSpinBox::valueChanged, this, &ScheduledShutdownWidget::updateTimeInfo);
    connect(minuteSpinBox, &QSpinBox::valueChanged, this, &ScheduledShutdownWidget::updateTimeInfo);
}

void ScheduledShutdownWidget::setupUI()
{
    QVBoxLayout *layout = new QVBoxLayout(this);
    layout->setContentsMargins(40, 40, 40, 40);
    layout->setSpacing(20);
    layout->setAlignment(Qt::AlignTop);

    // Title
    QLabel *titleLabel = new QLabel("Configure Shutdown", this);
    titleLabel->setStyleSheet("font-size: 24px; font-weight: 600; color: #333; border: none;");
    layout->addWidget(titleLabel);

    // Current Time Row
    QHBoxLayout *curTimeLayout = new QHBoxLayout();
    curTimeLayout->setSpacing(0);
    curTimeLayout->setAlignment(Qt::AlignLeft);

    QLabel *lblCurTitle = new QLabel("Current Time:", this);
    lblCurTitle->setFixedWidth(160);
    lblCurTitle->setStyleSheet("font-size: 15px; color: #333; margin-top: 5px;");
    curTimeLayout->addWidget(lblCurTitle);

    lblCurrentTimeValue = new QLabel(this);
    lblCurrentTimeValue->setStyleSheet("font-size: 15px; font-weight: bold; color: #0078d4; margin-top: 5px;");
    curTimeLayout->addWidget(lblCurrentTimeValue);
    layout->addLayout(curTimeLayout);

    QLabel *descLabel = new QLabel("Select duration:", this);
    descLabel->setStyleSheet("font-size: 14px; color: #666; border: none; margin-top: 5px;");
    layout->addWidget(descLabel);

    // Inputs
    QHBoxLayout *inputRowLayout = new QHBoxLayout();
    inputRowLayout->setSpacing(10);
    inputRowLayout->setAlignment(Qt::AlignLeft);

    QString unitLabelStyle = "font-size: 14px; color: #333; margin-right: 15px;";

    // --- Hour Input ---
    hourSpinBox = new QSpinBox(this);
    hourSpinBox->setRange(0, 23);
    hourSpinBox->setValue(4);

    SpinBoxWrapper* hourWrapper = new SpinBoxWrapper(hourSpinBox, this);
    inputRowLayout->addWidget(hourWrapper);

    QLabel *hrLabel = new QLabel("Hours", this);
    hrLabel->setStyleSheet(unitLabelStyle);
    inputRowLayout->addWidget(hrLabel);

    // --- Minute Input ---
    minuteSpinBox = new QSpinBox(this);
    minuteSpinBox->setRange(0, 59);

    SpinBoxWrapper* minWrapper = new SpinBoxWrapper(minuteSpinBox, this);
    inputRowLayout->addWidget(minWrapper);

    QLabel *minLabel = new QLabel("Minutes", this);
    minLabel->setStyleSheet(unitLabelStyle);
    inputRowLayout->addWidget(minLabel);
    layout->addLayout(inputRowLayout);

    // Estimated Time
    QHBoxLayout *estTimeLayout = new QHBoxLayout();
    estTimeLayout->setSpacing(0);
    estTimeLayout->setAlignment(Qt::AlignLeft);

    QLabel *lblEstTitle = new QLabel("Estimated Shutdown:", this);
    lblEstTitle->setFixedWidth(160);
    lblEstTitle->setStyleSheet("font-size: 15px; color: #333;");
    estTimeLayout->addWidget(lblEstTitle);

    lblEstTimeValue = new QLabel(this);
    lblEstTimeValue->setStyleSheet("font-size: 15px; font-weight: bold; color: #0078d4;");
    estTimeLayout->addWidget(lblEstTimeValue);
    layout->addLayout(estTimeLayout);

    layout->addSpacing(10);

    // Buttons
    QHBoxLayout *btnRowLayout = new QHBoxLayout();
    btnRowLayout->setSpacing(20);
    btnRowLayout->setAlignment(Qt::AlignLeft);

    QPushButton *btnSchedule = new QPushButton("Schedule Shutdown", this);
    btnSchedule->setCursor(Qt::PointingHandCursor);
    btnSchedule->setFixedWidth(180);
    btnSchedule->setStyleSheet("QPushButton { background-color: #0078d4; color: white; border: none; border-radius: 4px; padding: 10px 20px; font-size: 14px; font-weight: 600;} QPushButton:hover { background-color: #026ec1; }");
    connect(btnSchedule, &QPushButton::clicked, this, &ScheduledShutdownWidget::onScheduleClicked);
    btnRowLayout->addWidget(btnSchedule);

    QPushButton *btnCancel = new QPushButton("Cancel Plan", this);
    btnCancel->setCursor(Qt::PointingHandCursor);
    btnCancel->setFixedWidth(180);
    btnCancel->setStyleSheet("QPushButton { background-color: #e0e0e0; color: #333; border: none; border-radius: 4px; padding: 10px 20px; font-size: 14px; } QPushButton:hover { background-color: #d0d0d0; }");
    connect(btnCancel, &QPushButton::clicked, this, &ScheduledShutdownWidget::onCancelClicked);
    btnRowLayout->addWidget(btnCancel);

    layout->addLayout(btnRowLayout);
    layout->addStretch();
}

void ScheduledShutdownWidget::updateTimeInfo()
{
    QDateTime now = QDateTime::currentDateTime();
    lblCurrentTimeValue->setText(now.toString("ddd h:mm:ss AP"));

    int hours = hourSpinBox->value();
    int mins = minuteSpinBox->value();
    int totalSeconds = (hours * 3600) + (mins * 60);

    if (totalSeconds > 0) {
        lblEstTimeValue->setText(now.addSecs(totalSeconds).toString("ddd h:mm:ss AP"));
    } else {
        lblEstTimeValue->setText("--");
    }
}

// [UPDATED] Schedule with Feedback
void ScheduledShutdownWidget::onScheduleClicked()
{
    int hours = hourSpinBox->value();
    int mins = minuteSpinBox->value();
    int totalSeconds = (hours * 3600) + (mins * 60);

    if (totalSeconds <= 0) {
        QMessageBox::warning(this, "Invalid Time", "Please set a duration greater than 0.");
        return;
    }

    QProcess process;
    process.start("shutdown", QStringList() << "-s" << "-t" << QString::number(totalSeconds));
    process.waitForFinished();

    // Read both StdOut and StdErr as Windows puts messages in different streams depending on version/error
    QString output = QString::fromLocal8Bit(process.readAllStandardOutput());
    QString error = QString::fromLocal8Bit(process.readAllStandardError());
    QString allOutput = (output + error).trimmed();

    if (allOutput.isEmpty()) {
        QMessageBox::information(this, "Success", "Shutdown scheduled successfully.");
    } else if (allOutput.contains("already", Qt::CaseInsensitive)) {
        QMessageBox::warning(this, "Conflict", "A shutdown is already scheduled. Please cancel the existing plan first.");
    } else {
        QMessageBox::critical(this, "Error", "Temporarily unable to complete this function.\nDetails: " + allOutput);
    }
}

// [UPDATED] Cancel with Feedback
void ScheduledShutdownWidget::onCancelClicked()
{
    QProcess process;
    process.start("shutdown", QStringList() << "-a");
    process.waitForFinished();

    QString output = QString::fromLocal8Bit(process.readAllStandardOutput());
    QString error = QString::fromLocal8Bit(process.readAllStandardError());
    QString allOutput = (output + error).trimmed();

    if (allOutput.isEmpty()) {
        QMessageBox::information(this, "Success", "Scheduled shutdown cancelled successfully.");
    } else if (allOutput.contains("Unable", Qt::CaseInsensitive)) {
        // "Unable to abort the system shutdown because no shutdown was in progress."
        QMessageBox::information(this, "No Task", "No scheduled shutdown task found.");
    } else {
        QMessageBox::critical(this, "Error", "Temporarily unable to complete this function.\nDetails: " + allOutput);
    }
}
